<?php

namespace App\Http\Controllers\Api\V1;

use App\Http\Controllers\Controller;
use App\Http\Requests\StoreSettingRequest;
use App\Http\Requests\UpdateSettingRequest;
use App\Models\Setting;
use Illuminate\Http\Request;

class SettingController extends Controller
{
    // GET /api/v1/settings?search=foo&sort=-created_at&per_page=50
    public function index(Request $request)
    {
        $q = Setting::query()->where('created_by', $request->user()->id);

        if ($s = $request->string('search')->toString()) {
            $q->where('name', 'like', "%{$s}%");
        }

        if ($sort = $request->string('sort')->toString()) {
            $dir = str_starts_with($sort, '-') ? 'desc' : 'asc';
            $col = ltrim($sort, '-');
            $q->orderBy($col, $dir);
        } else {
            $q->latest('id');
        }

        $perPage = min(max((int)$request->query('per_page', 15), 1), 100);
        return response()->json($q->paginate($perPage));
    }

    // POST /api/v1/settings
    public function store(StoreSettingRequest $request)
    {
        $data = $request->validated();
        $data['created_by'] = $request->user()->id;

        $setting = Setting::create($data);
        return response()->json($setting, 201);
    }

    // GET /api/v1/settings/{setting}
    public function show(Request $request, Setting $setting)
    {
        if ($setting->created_by !== $request->user()->id) {
            return response()->json(['message' => 'Not found'], 404);
        }
        return response()->json($setting);
    }

    // PUT/PATCH /api/v1/settings/{setting}
    public function update(UpdateSettingRequest $request, Setting $setting)
    {
        if ($setting->created_by !== $request->user()->id) {
            return response()->json(['message' => 'Not found'], 404);
        }
        $setting->update($request->validated());
        return response()->json($setting);
    }

    // DELETE /api/v1/settings/{setting}
    public function destroy(Request $request, Setting $setting)
    {
        if ($setting->created_by !== $request->user()->id) {
            return response()->json(['message' => 'Not found'], 404);
        }
        $setting->delete();
        return response()->json(null, 204);
    }
}
