<?php

namespace App\Http\Controllers\Api\V1;

use App\Http\Controllers\Controller;
use App\Http\Requests\StoreDucumentUploadRequest;
use App\Http\Requests\UpdateDucumentUploadRequest;
use App\Models\DucumentUpload;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class DucumentUploadController extends Controller
{
    // GET /api/v1/ducument-uploads?search=profile&role=admin&sort=-created_at&per_page=25
    public function index(Request $request)
    {
        $q = DucumentUpload::query()->where('created_by', $request->user()->id);

        if ($s = $request->string('search')->toString()) {
            $q->where(function ($w) use ($s) {
                $w->where('name', 'like', "%{$s}%")
                  ->orWhere('description', 'like', "%{$s}%")
                  ->orWhere('role', 'like', "%{$s}%");
            });
        }
        if ($role = $request->string('role')->toString()) {
            $q->where('role', $role);
        }

        if ($sort = $request->string('sort')->toString()) {
            $dir = str_starts_with($sort, '-') ? 'desc' : 'asc';
            $col = ltrim($sort, '-');
            $q->orderBy($col, $dir);
        } else {
            $q->latest('id');
        }

        $perPage = min(max((int)$request->query('per_page', 15), 1), 100);
        return response()->json($q->paginate($perPage));
    }

    // POST /api/v1/ducument-uploads (multipart/form-data)
    public function store(StoreDucumentUploadRequest $request)
    {
        $path = $request->file('document')->store('docs', 'public');

        $doc = DucumentUpload::create([
            'name'        => $request->name,
            'role'        => $request->role,
            'description' => $request->input('description'),
            'document'    => $path,                // relative path on public disk
            'created_by'  => $request->user()->id, // owner
        ]);

        return response()->json($doc, 201);
    }

    // GET /api/v1/ducument-uploads/{ducument_upload}
    public function show(Request $request, DucumentUpload $ducument_upload)
    {
        if ($ducument_upload->created_by !== $request->user()->id) {
            return response()->json(['message' => 'Not found'], 404);
        }
        return response()->json($ducument_upload);
    }

    // PUT/PATCH /api/v1/ducument-uploads/{ducument_upload}
    public function update(UpdateDucumentUploadRequest $request, DucumentUpload $ducument_upload)
    {
        if ($ducument_upload->created_by !== $request->user()->id) {
            return response()->json(['message' => 'Not found'], 404);
        }

        $data = $request->validated();

        if ($request->hasFile('document')) {
            // delete old file if exists
            if ($ducument_upload->document && Storage::disk('public')->exists($ducument_upload->document)) {
                Storage::disk('public')->delete($ducument_upload->document);
            }
            $data['document'] = $request->file('document')->store('docs', 'public');
        }

        $ducument_upload->update($data);

        return response()->json($ducument_upload);
    }

    // DELETE /api/v1/ducument-uploads/{ducument_upload}
    public function destroy(Request $request, DucumentUpload $ducument_upload)
    {
        if ($ducument_upload->created_by !== $request->user()->id) {
            return response()->json(['message' => 'Not found'], 404);
        }

        if ($ducument_upload->document && Storage::disk('public')->exists($ducument_upload->document)) {
            Storage::disk('public')->delete($ducument_upload->document);
        }

        $ducument_upload->delete();
        return response()->json(null, 204);
    }
}
