<?php

namespace App\Http\Controllers\Api\V1;

use App\Http\Controllers\Controller;
use App\Http\Requests\StoreCustomerRequest;
use App\Http\Requests\UpdateCustomerRequest;
use App\Models\Customer;
use Illuminate\Http\Request;

class CustomerController extends Controller
{
    // GET /api/v1/customers?search=acme&filter[is_active]=1&sort=-created_at&per_page=25
    public function index(Request $request)
    {
        $q = Customer::query()->where('created_by', $request->user()->id);

        if ($s = $request->string('search')->toString()) {
            $q->where(function ($w) use ($s) {
                $w->where('name','like',"%{$s}%")
                  ->orWhere('email','like',"%{$s}%")
                  ->orWhere('contact','like',"%{$s}%")
                  ->orWhere('billing_name','like',"%{$s}%")
                  ->orWhere('shipping_name','like',"%{$s}%")
                  ->orWhere('billing_city','like',"%{$s}%")
                  ->orWhere('shipping_city','like',"%{$s}%");
            });
        }

        if ($filters = $request->input('filter', [])) {
            foreach (['is_active','billing_country','shipping_country'] as $col) {
                if (array_key_exists($col, $filters) && $filters[$col] !== '') {
                    $q->where($col, $filters[$col]);
                }
            }
        }

        if ($from = $request->date('from')) $q->whereDate('created_at','>=',$from);
        if ($to   = $request->date('to'))   $q->whereDate('created_at','<=',$to);

        if ($sort = $request->string('sort')->toString()) {
            $dir = str_starts_with($sort,'-') ? 'desc' : 'asc';
            $col = ltrim($sort,'-');
            $q->orderBy($col,$dir);
        } else {
            $q->latest('id');
        }

        $perPage = min(max((int)$request->query('per_page',15),1),100);
        return response()->json($q->paginate($perPage));
    }

    // POST /api/v1/customers
    public function store(StoreCustomerRequest $request)
    {
        $data = $request->validated();
        $data['created_by'] = $request->user()->id;
        $data['is_active']  = $data['is_active'] ?? 1;

        $customer = Customer::create($data);
        return response()->json($customer, 201);
    }

    // GET /api/v1/customers/{customer}
    public function show(Request $request, Customer $customer)
    {
        if ($customer->created_by !== $request->user()->id) {
            return response()->json(['message' => 'Not found'], 404);
        }
        return response()->json($customer);
    }

    // PUT/PATCH /api/v1/customers/{customer}
    public function update(UpdateCustomerRequest $request, Customer $customer)
    {
        if ($customer->created_by !== $request->user()->id) {
            return response()->json(['message' => 'Not found'], 404);
        }
        $customer->update($request->validated());
        return response()->json($customer);
    }

    // DELETE /api/v1/customers/{customer}
    // Logical delete -> set is_active = 0
    public function destroy(Request $request, Customer $customer)
    {
        if ($customer->created_by !== $request->user()->id) {
            return response()->json(['message' => 'Not found'], 404);
        }
        $customer->update(['is_active' => 0]);
        return response()->json(null, 204);
    }
}
